
import React, { useState, useEffect, useMemo } from 'react';
import * as Lucide from 'lucide-react';
import { fetchTickers, fetchKlines } from './services/bingx';
import { calculateRSI, analyzeTrend, calculateWicks, calculateAlgoScore } from './services/indicators';
import { MarketItem, FilterMode, SortMode } from './types';
import { getSniperVerdict } from './services/gemini';

const App = () => {
  const [db, setDb] = useState<Record<string, MarketItem>>({});
  const [favs, setFavs] = useState<Set<string>>(new Set(JSON.parse(localStorage.getItem('sniper_favs') || '[]')));
  const [filter, setFilter] = useState(FilterMode.ALL);
  const [sort, setSort] = useState(SortMode.ALGO_SCORE);
  const [loading, setLoading] = useState(true);
  const [progress, setProgress] = useState(0);
  const [selected, setSelected] = useState<string | null>(null);
  const [verdict, setVerdict] = useState("");

  const init = async () => {
    try {
      const tickers = await fetchTickers();
      const pairs = tickers.filter((t: any) => t.symbol.endsWith('-USDT')).slice(0, 40);
      const newDb: Record<string, MarketItem> = {};
      
      for (let i = 0; i < pairs.length; i++) {
        const t = pairs[i];
        const [k15, k1h, k4h] = await Promise.all([
          fetchKlines(t.symbol, '15m'),
          fetchKlines(t.symbol, '1h'),
          fetchKlines(t.symbol, '4h')
        ]);

        if (k1h.length > 0) {
          const rsi = calculateRSI(k1h);
          const wicks = calculateWicks(k1h[0]);
          const h1Trend = analyzeTrend(k1h);
          const h4Trend = analyzeTrend(k4h);
          
          newDb[t.symbol] = {
            symbol: t.symbol,
            price: parseFloat(t.lastPrice),
            prevPrice: parseFloat(t.lastPrice),
            change24h: parseFloat(t.priceChangePercent),
            volume24h: parseFloat(t.volume),
            rsi,
            wick: wicks,
            sr: { nearestRes: null, nearestSup: null, proximityRes: 0, proximitySup: 0 },
            mtf: {
              m15: { rsi: calculateRSI(k15), trend: analyzeTrend(k15) },
              h1: { rsi, trend: h1Trend },
              h4: { rsi: calculateRSI(k4h), trend: h4Trend }
            },
            algoScore: calculateAlgoScore(rsi, wicks, h1Trend, h4Trend),
            lastUpdate: Date.now()
          };
        }
        setProgress(Math.round(((i + 1) / pairs.length) * 100));
      }
      setDb(newDb);
      setLoading(false);
    } catch (e) { console.error(e); }
  };

  useEffect(() => { init(); }, []);

  const sortedData = useMemo(() => {
    let items = Object.values(db);
    if (filter === FilterMode.FAVORITES) items = items.filter(i => favs.has(i.symbol));
    if (filter === FilterMode.HIGH_SCORE) items = items.filter(i => i.algoScore > 70);
    if (filter === FilterMode.CONFLUENCE) items = items.filter(i => i.mtf.h1.trend === i.mtf.h4.trend);

    return items.sort((a, b) => {
      if (sort === SortMode.ALGO_SCORE) return b.algoScore - a.algoScore;
      if (sort === SortMode.VOLATILITY) return Math.abs(b.change24h) - Math.abs(a.change24h);
      return b.volume24h - a.volume24h;
    });
  }, [db, filter, sort, favs]);

  const toggleFav = (s: string) => {
    const n = new Set(favs);
    n.has(s) ? n.delete(s) : n.add(s);
    setFavs(n);
    localStorage.setItem('sniper_favs', JSON.stringify(Array.from(n)));
  };

  const runAnalysis = async (s: string) => {
    setSelected(s);
    setVerdict("Menghitung probabilitas...");
    const res = await getSniperVerdict(db[s]);
    setVerdict(res || "");
  };

  if (loading) {
    return (
      <div className="h-screen w-screen flex flex-col items-center justify-center bg-slate-950">
        <Lucide.Crosshair className="text-red-600 mb-6 animate-spin" size={48} />
        <h1 className="text-2xl font-black tracking-tighter text-white mb-2">BINGX PRO SNIPER</h1>
        <div className="w-64 h-1 bg-slate-900 rounded-full overflow-hidden">
          <div className="h-full bg-red-600 transition-all duration-300" style={{ width: `${progress}%` }} />
        </div>
        <p className="text-[10px] text-slate-500 mt-4 uppercase tracking-widest font-bold">Scanning Global Liquidity: {progress}%</p>
      </div>
    );
  }

  return (
    <div className="min-h-screen p-6 max-w-[1600px] mx-auto">
      {/* Header */}
      <div className="flex flex-col md:flex-row justify-between items-center mb-10 gap-6">
        <div className="flex items-center gap-4">
          <div className="bg-red-600/10 p-3 rounded-2xl border border-red-600/20">
            <Lucide.Zap className="text-red-600" size={28} />
          </div>
          <div>
            <h1 className="text-2xl font-black text-white tracking-tight leading-none">PRO SNIPER <span className="text-red-600">ELITE</span></h1>
            <p className="text-[9px] text-slate-500 uppercase font-black tracking-[0.3em] mt-1">Multi-Timeframe Engine Active</p>
          </div>
        </div>

        <div className="flex flex-wrap gap-2">
          <select 
            className="bg-slate-900 border border-slate-800 text-xs font-bold px-4 py-2 rounded-xl focus:outline-none"
            value={filter} onChange={e => setFilter(e.target.value as FilterMode)}
          >
            <option value={FilterMode.ALL}>Show All</option>
            <option value={FilterMode.CONFLUENCE}>Trend Confluence</option>
            <option value={FilterMode.HIGH_SCORE}>Algo > 70</option>
            <option value={FilterMode.FAVORITES}>Favorites</option>
          </select>
          <select 
            className="bg-slate-900 border border-slate-800 text-xs font-bold px-4 py-2 rounded-xl focus:outline-none"
            value={sort} onChange={e => setSort(e.target.value as SortMode)}
          >
            <option value={SortMode.ALGO_SCORE}>Sort by Score</option>
            <option value={SortMode.VOLATILITY}>Sort by Volatility</option>
            <option value={SortMode.VOLUME}>Sort by Volume</option>
          </select>
          <button onClick={init} className="bg-slate-900 p-2 rounded-xl border border-slate-800 hover:text-red-500 transition-colors">
            <Lucide.RefreshCw size={18} />
          </button>
        </div>
      </div>

      {/* Grid */}
      <div className="grid grid-cols-1 xl:grid-cols-4 gap-6 mb-8">
        <div className="xl:col-span-3 bg-slate-900/50 border border-slate-800/50 rounded-3xl overflow-hidden backdrop-blur-md">
          <table className="w-full text-left">
            <thead className="bg-slate-950/50 text-[10px] text-slate-500 font-black uppercase tracking-widest border-b border-slate-800">
              <tr>
                <th className="px-6 py-5 w-12">Fav</th>
                <th className="px-6 py-5">Pair</th>
                <th className="px-6 py-5">Price</th>
                <th className="px-6 py-5">Score</th>
                <th className="px-6 py-5">RSI (1H)</th>
                <th className="px-6 py-5">Wick Rej</th>
                <th className="px-6 py-5 text-right">Action</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-slate-800/30">
              {sortedData.map(i => (
                <tr key={i.symbol} className="hover:bg-red-600/[0.02] group transition-colors">
                  <td className="px-6 py-4">
                    <button onClick={() => toggleFav(i.symbol)} className={`${favs.has(i.symbol) ? 'text-yellow-500' : 'text-slate-700'}`}>
                      <Lucide.Star size={16} fill={favs.has(i.symbol) ? 'currentColor' : 'none'} />
                    </button>
                  </td>
                  <td className="px-6 py-4">
                    <span className="text-sm font-black text-white">{i.symbol.replace('-USDT', '')}</span>
                  </td>
                  <td className="px-6 py-4">
                    <span className="text-sm font-black mono text-slate-300">{i.price.toLocaleString()}</span>
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex items-center gap-3">
                      <span className={`text-sm font-black ${i.algoScore > 75 ? 'text-red-500 animate-pulse-red' : i.algoScore < 25 ? 'text-emerald-500' : 'text-slate-400'}`}>
                        {Math.round(i.algoScore)}
                      </span>
                      <div className="w-16 h-1 bg-slate-800 rounded-full overflow-hidden hidden sm:block">
                        <div className={`h-full ${i.algoScore > 75 ? 'bg-red-500' : 'bg-slate-600'}`} style={{ width: `${i.algoScore}%` }} />
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4">
                    <span className={`text-xs font-bold px-2 py-0.5 rounded ${i.rsi > 70 ? 'bg-red-500/10 text-red-500' : i.rsi < 30 ? 'bg-emerald-500/10 text-emerald-500' : 'text-slate-500'}`}>
                      {i.rsi.toFixed(1)}
                    </span>
                  </td>
                  <td className="px-6 py-4">
                    <span className={`text-xs font-bold ${i.wick.upperPct > 0.4 ? 'text-red-500' : 'text-slate-600'}`}>
                      {i.wick.upperPct.toFixed(2)}%
                    </span>
                  </td>
                  <td className="px-6 py-4 text-right">
                    <button onClick={() => runAnalysis(i.symbol)} className="text-slate-500 hover:text-white bg-slate-800/50 p-2 rounded-lg transition-all">
                      <Lucide.Crosshair size={16} />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Sidebar Analysis */}
        <div className="space-y-6">
          <div className="bg-slate-900 border border-slate-800 rounded-3xl p-6">
            <h3 className="text-xs font-black text-slate-500 uppercase tracking-widest mb-6">Market Intel</h3>
            <div className="grid grid-cols-2 gap-4">
              <div className="bg-slate-950 p-4 rounded-2xl border border-slate-800">
                <p className="text-[9px] text-slate-600 font-bold uppercase mb-1">Pairs</p>
                <p className="text-xl font-black text-white">{Object.keys(db).length}</p>
              </div>
              <div className="bg-slate-950 p-4 rounded-2xl border border-slate-800">
                <p className="text-[9px] text-slate-600 font-bold uppercase mb-1">Opportunities</p>
                <p className="text-xl font-black text-emerald-500">{sortedData.filter(d => d.algoScore > 70).length}</p>
              </div>
            </div>
          </div>

          <div className={`bg-slate-900 border ${selected ? 'border-red-600/50 shadow-[0_0_30px_rgba(220,38,38,0.1)]' : 'border-slate-800'} rounded-3xl p-6 transition-all`}>
            <div className="flex items-center gap-3 mb-6">
              <Lucide.Cpu size={18} className="text-red-500" />
              <h3 className="text-xs font-black text-white uppercase tracking-widest">AI Analyst</h3>
            </div>
            {selected ? (
              <div className="animate-in fade-in duration-500">
                <div className="flex justify-between mb-4">
                  <span className="text-sm font-black text-white">{selected.replace('-USDT', '')}</span>
                  <span className="text-[10px] bg-red-600/10 text-red-500 px-2 py-0.5 rounded font-black">ACTIVE SCAN</span>
                </div>
                <p className="text-xs text-slate-400 leading-relaxed font-medium">
                  {verdict}
                </p>
                <button 
                  onClick={() => window.open(`https://www.tradingview.com/chart/?symbol=BINGX:${selected.replace('-','')}.P`, '_blank')}
                  className="w-full mt-6 py-3 bg-red-600 text-white rounded-xl text-[10px] font-black uppercase tracking-widest shadow-lg shadow-red-600/20"
                >
                  View Live Chart
                </button>
              </div>
            ) : (
              <p className="text-xs text-slate-600 font-bold text-center py-10">Select a target for deep AI analysis</p>
            )}
          </div>
        </div>
      </div>

      {/* Footer Status */}
      <footer className="fixed bottom-0 left-0 right-0 bg-slate-950/80 backdrop-blur-md border-t border-slate-800/50 px-6 py-3">
        <div className="max-w-[1600px] mx-auto flex justify-between items-center">
          <div className="flex items-center gap-4">
            <div className="flex items-center gap-2">
              <div className="w-1.5 h-1.5 bg-emerald-500 rounded-full animate-pulse" />
              <span className="text-[9px] font-black text-slate-500 uppercase tracking-widest">System Online</span>
            </div>
            <span className="text-[9px] font-black text-slate-600 uppercase tracking-widest hidden sm:inline">| Latency: 42ms</span>
          </div>
          <span className="text-[9px] font-black text-slate-500 uppercase tracking-widest">© 2024 Pro Sniper Elite</span>
        </div>
      </footer>
    </div>
  );
};

export default App;
