
import { Kline, WickMetrics, SupportResistance, TimeframeAnalysis } from '../types';

export const calculateRSI = (klines: Kline[], period: number = 14): number => {
  if (klines.length < period + 1) return 50;
  let gains = 0, losses = 0;
  for (let i = klines.length - period; i < klines.length; i++) {
    const diff = klines[i].close - klines[i - 1].close;
    if (diff > 0) gains += diff; else losses -= diff;
  }
  if (losses === 0) return 100;
  const rs = (gains / period) / (losses / period);
  return 100 - (100 / (1 + rs));
};

export const calculateEMA = (klines: Kline[], period: number): number => {
  const k = 2 / (period + 1);
  let ema = klines[0].close;
  for (let i = 1; i < klines.length; i++) {
    ema = klines[i].close * k + ema * (1 - k);
  }
  return ema;
};

export const analyzeTrend = (klines: Kline[]): 'bull' | 'bear' | 'side' => {
  if (klines.length < 50) return 'side';
  const ema20 = calculateEMA(klines.slice(-20), 20);
  const ema50 = calculateEMA(klines.slice(-50), 50);
  const price = klines[klines.length - 1].close;
  if (price > ema20 && ema20 > ema50) return 'bull';
  if (price < ema20 && ema20 < ema50) return 'bear';
  return 'side';
};

export const calculateWicks = (k: Kline): WickMetrics => {
  const bodyTop = Math.max(k.open, k.close);
  const bodyBottom = Math.min(k.open, k.close);
  const upper = ((k.high - bodyTop) / k.close) * 100;
  const lower = ((bodyBottom - k.low) / k.close) * 100;
  return { upperPct: upper, lowerPct: lower, isStrongRejection: upper > 0.5 || lower > 0.5 };
};

export const calculateAlgoScore = (rsi: number, wicks: WickMetrics, trend: string, mtfTrend: string): number => {
  let score = 50;
  // RSI Overbought bias for Short
  if (rsi > 75) score += 20;
  else if (rsi < 25) score -= 20;

  // Rejection bias
  if (wicks.upperPct > 0.6) score += 15;
  if (wicks.lowerPct > 0.6) score -= 15;

  // Trend Confluence
  if (trend === 'bear' && mtfTrend === 'bear') score += 15;
  if (trend === 'bull' && mtfTrend === 'bull') score -= 15;

  return Math.min(100, Math.max(0, score));
};
